<?php
/**
 * Plugin Name: پیامک کیف پول ووکامرس - نسخه نهایی
 * Description: ارسال پیامک با کاوه نگار هنگام تغییر موجودی کیف پول - با ردیابی تغییرات
 * Version: 3.0.0
 * Author: شما
 */

defined('ABSPATH') || exit;

// ==================== فعال‌سازی سیستم دیباگ ====================
define('WALLET_SMS_DEBUG', true);

function wallet_sms_log($message, $data = null) {
    if (WALLET_SMS_DEBUG) {
        $log_message = '[' . current_time('Y-m-d H:i:s') . '] ' . $message;
        if ($data !== null) {
            $log_message .= ' | Data: ' . print_r($data, true);
        }
        error_log('WALLET SMS: ' . $log_message);
    }
}

// ==================== ذخیره مقادیر قبلی برای مقایسه ====================
add_action('update_user_meta', 'wallet_sms_save_old_balance', 1, 4);
function wallet_sms_save_old_balance($meta_id, $user_id, $meta_key, $meta_value) {
    if ($meta_key === '_current_woo_wallet_balance') {
        $old_value = get_user_meta($user_id, $meta_key, true);
        if ($old_value !== '') {
            $GLOBALS['wallet_old_balance'] = $old_value;
            $GLOBALS['wallet_user_id'] = $user_id;
            wallet_sms_log('💾 مقدار قدیم ذخیره شد', [
                'user_id' => $user_id,
                'old_balance' => $old_value,
                'new_balance' => $meta_value
            ]);
        }
    }
}

// ==================== هوک اصلی برای تشخیص تغییر موجودی ====================
add_action('updated_user_meta', 'wallet_sms_detect_balance_change', 10, 4);
function wallet_sms_detect_balance_change($meta_id, $user_id, $meta_key, $meta_value) {
    // فقط برای متای کیف پول
    if ($meta_key !== '_current_woo_wallet_balance') {
        return;
    }
    
    wallet_sms_log('🔍 تغییر در متای کیف پول شناسایی شد', [
        'user_id' => $user_id,
        'meta_key' => $meta_key,
        'new_value' => $meta_value
    ]);
    
    // دریافت مقدار قدیمی
    $old_balance = isset($GLOBALS['wallet_old_balance']) ? $GLOBALS['wallet_old_balance'] : 0;
    
    // اگر مقدار قدیمی نداریم (مثلاً اولین بار است)
    if ($old_balance === 0) {
        $old_balance = get_user_meta($user_id, $meta_key, true);
        if ($old_balance === $meta_value) {
            wallet_sms_log('⚠️ تغییر واقعی تشخیص داده نشد (مقادیر برابر)');
            return;
        }
    }
    
    // محاسبه تغییرات
    $change_amount = floatval($meta_value) - floatval($old_balance);
    
    // اگر تغییر قابل توجهی نیست
    if (abs($change_amount) < 1) {
        wallet_sms_log('⚠️ تغییر ناچیز است - پیامک ارسال نمی‌شود', $change_amount);
        return;
    }
    
    // تشخیص نوع تغییر
    $type = ($change_amount > 0) ? 'credit' : 'debit';
    $amount = abs($change_amount);
    
    wallet_sms_log('💰 تغییر موجودی محاسبه شد', [
        'old_balance' => $old_balance,
        'new_balance' => $meta_value,
        'change_amount' => $change_amount,
        'type' => $type,
        'absolute_amount' => $amount
    ]);
    
    // ارسال پیامک
    wallet_sms_send_notification($user_id, $amount, $type, $meta_value);
}

// ==================== هوک‌های جایگزین برای اطمینان ====================

// هوک استاندارد کیف پول
add_action('woocommerce_wallet_balance_updated', function($user_id, $amount, $type) {
    wallet_sms_log('🎯 هوک اصلی کیف پول فعال شد', compact('user_id', 'amount', 'type'));
    
    // دریافت موجودی فعلی
    $balance = 0;
    if (function_exists('woo_wallet') && is_object(woo_wallet()->wallet)) {
        $balance = woo_wallet()->wallet->get_wallet_balance($user_id, 'edit');
    } else {
        $balance = get_user_meta($user_id, '_current_woo_wallet_balance', true);
    }
    
    wallet_sms_send_notification($user_id, $amount, $type, $balance);
}, 10, 3);

// هوک تراکنش‌ها
add_action('woocommerce_wallet_transaction_recorded', function($transaction_id, $user_id, $amount, $type, $details) {
    wallet_sms_log('💳 هوک تراکنش کیف پول فعال شد', [
        'transaction_id' => $transaction_id,
        'user_id' => $user_id,
        'amount' => $amount,
        'type' => $type
    ]);
    
    if (!in_array($type, ['credit', 'debit'])) {
        return;
    }
    
    $balance = get_user_meta($user_id, '_current_woo_wallet_balance', true);
    wallet_sms_send_notification($user_id, $amount, $type, $balance);
}, 10, 5);

// ==================== تابع اصلی ارسال پیامک ====================
function wallet_sms_send_notification($user_id, $amount, $type, $current_balance = null) {
    wallet_sms_log('🚀 شروع ارسال پیامک', [
        'user_id' => $user_id,
        'amount' => $amount,
        'type' => $type,
        'current_balance' => $current_balance
    ]);
    
    // بررسی فعال بودن سیستم
    $enabled = get_option('wallet_sms_enabled', 'no');
    if ($enabled !== 'yes') {
        wallet_sms_log('⏸️ سیستم پیامک غیرفعال است');
        return false;
    }
    
    // دریافت اطلاعات کاربر
    $user = get_user_by('id', $user_id);
    if (!$user) {
        wallet_sms_log('❌ کاربر یافت نشد', $user_id);
        return false;
    }
    
    $phone_number = $user->user_login;
    wallet_sms_log('📱 شماره کاربر', $phone_number);
    
    // بررسی معتبر بودن شماره
    if (!preg_match('/^09[0-9]{9}$/', $phone_number)) {
        wallet_sms_log('❌ شماره تلفن نامعتبر است', $phone_number);
        return false;
    }
    
    // اگر موجودی فعلی داده نشده، آن را دریافت کن
    if ($current_balance === null) {
        $current_balance = get_user_meta($user_id, '_current_woo_wallet_balance', true);
    }
    
    // تنظیم متن پیامک
    if ($type == 'credit') {
        $template = get_option('wallet_sms_credit_msg', '💵 افزایش موجودی
مبلغ: {amount} ریال
موجودی جدید: {balance} ریال
زمان: {date}');
    } else {
        $template = get_option('wallet_sms_debit_msg', '📉 کاهش موجودی  
مبلغ: {amount} ریال  
موجودی جدید: {balance} ریال  
زمان: {date}');
    }
    
    // جایگزینی متغیرها
    $message = str_replace(
        ['{amount}', '{balance}', '{date}', '{username}'],
        [
            number_format($amount),
            number_format($current_balance),
            current_time('Y/m/d H:i'),
            $user->display_name
        ],
        $template
    );
    
    wallet_sms_log('✉️ متن پیامک آماده', $message);
    
    // دریافت تنظیمات API
    $api_key = trim(get_option('wallet_sms_api_key', ''));
    $sender = trim(get_option('wallet_sms_sender', '10004346'));
    
    if (empty($api_key)) {
        wallet_sms_log('❌ API Key تنظیم نشده است');
        return false;
    }
    
    // ارسال پیامک به کاوه نگار
    return wallet_sms_send_to_kavenegar($phone_number, $message, $api_key, $sender, [
        'user_id' => $user_id,
        'amount' => $amount,
        'type' => $type
    ]);
}

// ==================== تابع ارسال به کاوه نگار ====================
function wallet_sms_send_to_kavenegar($phone, $message, $api_key, $sender, $context = []) {
    $url = "https://api.kavenegar.com/v1/{$api_key}/sms/send.json";
    
    $args = [
        'receptor' => $phone,
        'sender'   => $sender,
        'message'  => $message
    ];
    
    wallet_sms_log('📡 در حال ارسال به کاوه نگار', [
        'url' => str_replace($api_key, '***API_KEY***', $url),
        'receptor' => $phone,
        'sender' => $sender,
        'message_length' => strlen($message)
    ]);
    
    // ارسال با cURL برای کنترل بهتر
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($args));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    // تحلیل پاسخ
    $success = false;
    $response_data = null;
    
    if ($error) {
        wallet_sms_log('❌ خطای cURL', $error);
    } else {
        wallet_sms_log('📥 پاسخ دریافتی', [
            'http_code' => $http_code,
            'response' => $response
        ]);
        
        $response_data = json_decode($response, true);
        
        if ($http_code == 200 && isset($response_data['return']['status']) && $response_data['return']['status'] == 200) {
            $success = true;
            wallet_sms_log('✅ پیامک با موفقیت ارسال شد', [
                'message_id' => $response_data['entries'][0]['messageid'] ?? 'N/A',
                'cost' => $response_data['entries'][0]['cost'] ?? 'N/A'
            ]);
        } else {
            wallet_sms_log('❌ خطا در پاسخ کاوه نگار', $response_data);
        }
    }
    
    // ذخیره لاگ ارسال
    $log_entry = [
        'time' => current_time('mysql'),
        'phone' => $phone,
        'message' => $message,
        'success' => $success,
        'http_code' => $http_code,
        'response' => $response_data,
        'context' => $context
    ];
    
    update_option('wallet_sms_last_send', $log_entry);
    
    // ذخیره در تاریخچه (آخرین ۵۰ ارسال)
    $history = get_option('wallet_sms_history', []);
    array_unshift($history, $log_entry);
    $history = array_slice($history, 0, 50);
    update_option('wallet_sms_history', $history);
    
    return $success;
}

// ==================== تست دستی ====================
function wallet_sms_manual_test($user_id, $amount = 10000, $type = 'credit') {
    wallet_sms_log('🧪 TEST: شروع تست دستی', compact('user_id', 'amount', $type));
    
    $user = get_user_by('id', $user_id);
    if (!$user) {
        return '❌ کاربر یافت نشد';
    }
    
    $balance = get_user_meta($user_id, '_current_woo_wallet_balance', true);
    $result = wallet_sms_send_notification($user_id, $amount, $type, $balance);
    
    return $result ? '✅ تست موفقیت‌آمیز بود' : '❌ تست ناموفق بود';
}

// ==================== صفحه مدیریت ====================
add_action('admin_menu', 'wallet_sms_admin_menu');

function wallet_sms_admin_menu() {
    add_menu_page(
        'پیامک کیف پول',
        'پیامک کیف پول',
        'manage_options',
        'wallet-sms-settings',
        'wallet_sms_settings_page',
        'dashicons-email-alt',
        56
    );
    
    add_submenu_page(
        'wallet-sms-settings',
        'تست و دیباگ',
        'تست و دیباگ',
        'manage_options',
        'wallet-sms-debug',
        'wallet_sms_debug_page'
    );
    
    add_submenu_page(
        'wallet-sms-settings',
        'تاریخچه ارسال',
        'تاریخچه',
        'manage_options',
        'wallet-sms-history',
        'wallet_sms_history_page'
    );
}

function wallet_sms_settings_page() {
    if (isset($_POST['submit']) && wp_verify_nonce($_POST['_wpnonce'], 'wallet_sms_settings')) {
        update_option('wallet_sms_enabled', sanitize_text_field($_POST['enabled']));
        update_option('wallet_sms_api_key', sanitize_text_field($_POST['api_key']));
        update_option('wallet_sms_sender', sanitize_text_field($_POST['sender']));
        update_option('wallet_sms_credit_msg', sanitize_textarea_field($_POST['credit_msg']));
        update_option('wallet_sms_debit_msg', sanitize_textarea_field($_POST['debit_msg']));
        
        echo '<div class="notice notice-success is-dismissible"><p>✅ تنظیمات ذخیره شد.</p></div>';
    }
    
    $enabled = get_option('wallet_sms_enabled', 'no');
    $api_key = get_option('wallet_sms_api_key', '');
    $sender = get_option('wallet_sms_sender', '10004346');
    $credit_msg = get_option('wallet_sms_credit_msg', '💵 افزایش موجودی
مبلغ: {amount} ریال
موجودی جدید: {balance} ریال
زمان: {date}');
    $debit_msg = get_option('wallet_sms_debit_msg', '📉 کاهش موجودی  
مبلغ: {amount} ریال  
موجودی جدید: {balance} ریال  
زمان: {date}');
    ?>
    
    <div class="wrap">
        <h1><span class="dashicons dashicons-email-alt"></span> تنظیمات پیامک کیف پول</h1>
        
        <div class="card" style="max-width: 800px; margin-bottom: 20px;">
            <h3>📊 وضعیت سیستم</h3>
            <p>
                <strong>حالت دیباگ:</strong> 
                <span style="color: green; font-weight: bold;">فعال ✅</span> |
                <strong>آخرین تغییر:</strong> 
                <?php
                $last_log = get_option('wallet_sms_last_send', []);
                echo $last_log ? date_i18n('Y/m/d H:i', strtotime($last_log['time'])) : 'ثبت نشده';
                ?>
            </p>
        </div>
        
        <form method="post" action="" class="card" style="max-width: 800px;">
            <h3>⚙️ تنظیمات اصلی</h3>
            <?php wp_nonce_field('wallet_sms_settings'); ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">فعال‌سازی سیستم</th>
                    <td>
                        <label><input type="radio" name="enabled" value="yes" <?php checked($enabled, 'yes'); ?>> فعال</label>
                        <label style="margin-right: 20px;"><input type="radio" name="enabled" value="no" <?php checked($enabled, 'no'); ?>> غیرفعال</label>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><label for="api_key">API Key کاوه نگار</label></th>
                    <td>
                        <input type="text" id="api_key" name="api_key" value="<?php echo esc_attr($api_key); ?>" 
                               class="regular-text" style="font-family: monospace;" placeholder="مثال: 486B6D6A6E4E7733...">
                        <p class="description">از <a href="https://panel.kavenegar.com/Client/setting/index" target="_blank">پنل کاوه نگار → تنظیمات → کلید API</a> دریافت کنید</p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><label for="sender">شماره فرستنده</label></th>
                    <td>
                        <input type="text" id="sender" name="sender" value="<?php echo esc_attr($sender); ?>" 
                               class="regular-text" placeholder="10004346">
                        <p class="description">شماره خط پیامکی شما در کاوه نگار</p>
                    </td>
                </tr>
            </table>
            
            <h3>📝 متن پیامک‌ها</h3>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="credit_msg">پیام افزایش موجودی</label></th>
                    <td>
                        <textarea id="credit_msg" name="credit_msg" rows="4" class="large-text"><?php echo esc_textarea($credit_msg); ?></textarea>
                        <p class="description">
                            متغیرهای قابل استفاده: 
                            <code>{amount}</code> <code>{balance}</code> <code>{date}</code> <code>{username}</code>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><label for="debit_msg">پیام کاهش موجودی</label></th>
                    <td>
                        <textarea id="debit_msg" name="debit_msg" rows="4" class="large-text"><?php echo esc_textarea($debit_msg); ?></textarea>
                    </td>
                </tr>
            </table>
            
            <p class="submit">
                <button type="submit" name="submit" class="button button-primary button-large">
                    💾 ذخیره تنظیمات
                </button>
                <a href="<?php echo admin_url('admin.php?page=wallet-sms-debug'); ?>" class="button">
                    🔧 صفحه دیباگ
                </a>
            </p>
        </form>
    </div>
    <?php
}

function wallet_sms_debug_page() {
    // تست دستی
    $test_result = '';
    if (isset($_POST['manual_test']) && wp_verify_nonce($_POST['_wpnonce'], 'wallet_sms_test')) {
        $test_user_id = intval($_POST['test_user_id']);
        $test_amount = intval($_POST['test_amount']);
        $test_type = sanitize_text_field($_POST['test_type']);
        
        if ($test_user_id && $test_amount) {
            $test_result = wallet_sms_manual_test($test_user_id, $test_amount, $test_type);
        }
    }
    
    $last_send = get_option('wallet_sms_last_send', []);
    ?>
    
    <div class="wrap">
        <h1>🔧 تست و دیباگ سیستم</h1>
        
        <?php if ($test_result): ?>
            <div class="notice notice-<?php echo strpos($test_result, '✅') !== false ? 'success' : 'error'; ?>">
                <p><?php echo $test_result; ?></p>
            </div>
        <?php endif; ?>
        
        <div class="card" style="max-width: 800px; margin-bottom: 20px;">
            <h3>📤 آخرین ارسال</h3>
            <?php if (!empty($last_send)): ?>
                <table class="widefat striped">
                    <tr><th>زمان</th><td><?php echo date_i18n('Y/m/d H:i:s', strtotime($last_send['time'])); ?></td></tr>
                    <tr><th>شماره</th><td><?php echo esc_html($last_send['phone']); ?></td></tr>
                    <tr><th>وضعیت</th><td>
                        <?php if ($last_send['success']): ?>
                            <span style="color: green; font-weight: bold;">✅ موفق</span>
                        <?php else: ?>
                            <span style="color: red; font-weight: bold;">❌ ناموفق</span>
                        <?php endif; ?>
                    </td></tr>
                    <tr><th>کد HTTP</th><td><?php echo esc_html($last_send['http_code']); ?></td></tr>
                    <tr><th>پاسخ API</th><td><pre style="max-height: 200px; overflow: auto;"><?php 
                        echo esc_html(print_r($last_send['response'], true)); 
                    ?></pre></td></tr>
                </table>
            <?php else: ?>
                <p>هنوز ارسالی ثبت نشده است.</p>
            <?php endif; ?>
        </div>
        
        <div class="card" style="max-width: 800px;">
            <h3>🧪 تست دستی ارسال</h3>
            <form method="post" action="">
                <?php wp_nonce_field('wallet_sms_test'); ?>
                
                <table class="form-table">
                    <tr>
                        <th><label for="test_user_id">ID کاربر</label></th>
                        <td>
                            <input type="number" id="test_user_id" name="test_user_id" class="regular-text" required>
                            <p class="description">شناسه کاربری که می‌خواهید پیامک دریافت کند</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="test_amount">مبلغ (ریال)</label></th>
                        <td>
                            <input type="number" id="test_amount" name="test_amount" value="10000" class="regular-text" required>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="test_type">نوع تغییر</label></th>
                        <td>
                            <select id="test_type" name="test_type" class="regular-text">
                                <option value="credit">افزایش موجودی (credit)</option>
                                <option value="debit">کاهش موجودی (debit)</option>
                            </select>
                        </td>
                    </tr>
                </table>
                
                <p>
                    <button type="submit" name="manual_test" class="button button-primary">
                        🚀 اجرای تست ارسال پیامک
                    </button>
                    <span class="description">این تست مستقیماً به کاوه نگار درخواست می‌فرستد</span>
                </p>
            </form>
        </div>
        
        <div class="card" style="max-width: 800px; margin-top: 20px;">
            <h3>📋 راهنمای عیب‌یابی</h3>
            <ol>
                <li><strong>API Key را بررسی کنید:</strong> از صحت کلید API در پنل کاوه نگار مطمئن شوید</li>
                <li><strong>اعتبار حساب:</strong> موجودی حساب کاوه نگار خود را چک کنید</li>
                <li><strong>لاگ‌ها:</strong> فایل <code>wp-content/debug.log</code> را بررسی کنید</li>
                <li><strong>تست مستقیم:</strong> از بخش تست دستی بالا استفاده کنید</li>
                <li><strong>شماره فرستنده:</strong> شماره فرستنده باید در پنل کاوه نگار تأیید شده باشد</li>
            </ol>
        </div>
    </div>
    <?php
}

function wallet_sms_history_page() {
    $history = get_option('wallet_sms_history', []);
    ?>
    
    <div class="wrap">
        <h1>📜 تاریخچه ارسال پیامک‌ها</h1>
        
        <?php if (empty($history)): ?>
            <div class="card">
                <p>هیچ تاریخی از ارسال پیامک ثبت نشده است.</p>
            </div>
        <?php else: ?>
            <div class="card">
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>زمان</th>
                            <th>شماره</th>
                            <th>وضعیت</th>
                            <th>کد HTTP</th>
                            <th>جزئیات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($history as $index => $entry): ?>
                            <tr>
                                <td><?php echo date_i18n('Y/m/d H:i', strtotime($entry['time'])); ?></td>
                                <td><?php echo esc_html($entry['phone']); ?></td>
                                <td>
                                    <?php if ($entry['success']): ?>
                                        <span style="color: green;">✅</span>
                                    <?php else: ?>
                                        <span style="color: red;">❌</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo esc_html($entry['http_code']); ?></td>
                                <td>
                                    <button class="button button-small" onclick="alert('<?php echo esc_js(print_r($entry['response'], true)); ?>')">
                                        مشاهده پاسخ
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
    <?php
}

// لینک‌های سریع در صفحه افزونه‌ها
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    array_unshift($links, 
        '<a href="' . admin_url('admin.php?page=wallet-sms-debug') . '" style="color: #46b450; font-weight: bold;">تست</a>',
        '<a href="' . admin_url('admin.php?page=wallet-sms-settings') . '">تنظیمات</a>'
    );
    return $links;
});

// فعال‌سازی اولیه
register_activation_hook(__FILE__, function() {
    if (get_option('wallet_sms_enabled') === false) {
        update_option('wallet_sms_enabled', 'yes');
    }
    if (get_option('wallet_sms_sender') === false) {
        update_option('wallet_sms_sender', '10004346');
    }
});